/*
 *  Routines for Trident 4DWave NX/DX soundcards - Synthesizer
 *  Copyright (c) by Scott McNab <jedi@tartarus.uwa.edu.au>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../../include/driver.h"
#include "../../include/trident.h"
#include "../../include/seq_device.h"

EXPORT_NO_SYMBOLS;

/* linear to log volume conversion table (5.3 attenuation format) */
static unsigned int volume_table[255] = {
	16149, 15919, 15691, 15467, 15246, 15028, 14813,
	14602, 14393, 14187, 13985, 13785, 13588, 13394, 13202,
	13014, 12828, 12645, 12464, 12286, 12110, 11937, 11767,
	11598, 11433, 11269, 11108, 10950, 10793, 10639, 10487,
	10337, 10189, 10044, 9900, 9759, 9619, 9482, 9346, 
	9213, 9081, 8951, 8824, 8698, 8573, 8451, 8330,
	8211, 8094, 7978, 7864, 7752, 7641, 7532, 7424,
	7318, 7213, 7110, 7009, 6909, 6810, 6713, 6617,
	6522, 6429, 6337, 6246, 6157, 6069, 5983, 5897,
	5813, 5730, 5648, 5567, 5488, 5409, 5332, 5256,
	5181, 5107, 5034, 4962, 4891, 4821, 4752, 4684,
	4617, 4551, 4486, 4422, 4359, 4297, 4235, 4175,
	4115, 4056, 3998, 3941, 3885, 3829, 3775, 3721,
	3667, 3615, 3563, 3512, 3462, 3413, 3364, 3316,
	3269, 3222, 3176, 3130, 3086, 3042, 2998, 2955,
	2913, 2871, 2830, 2790, 2750, 2711, 2672, 2634,
	2596, 2559, 2523, 2486, 2451, 2416, 2381, 2347,
	2314, 2281, 2248, 2216, 2184, 2153, 2122, 2092,
	2062, 2033, 2004, 1975, 1947, 1919, 1891, 1864,
	1838, 1812, 1786, 1760, 1735, 1710, 1686, 1662,
	1638, 1614, 1591, 1569, 1546, 1524, 1502, 1481,
	1460, 1439, 1418, 1398, 1378, 1358, 1339, 1320,
	1301, 1282, 1264, 1246, 1228, 1211, 1193, 1176,
	1159, 1143, 1126, 1110, 1095, 1079, 1063, 1048,
	1033, 1018, 1004, 990, 975, 961, 948, 934,
	921, 908, 895, 882, 869, 857, 845, 833, 
	821, 809, 797, 786, 775, 764, 753, 742, 
	731, 721, 711, 700, 690, 681, 671, 661, 
	652, 642, 633, 624, 615, 606, 598, 589, 
	581, 573, 564, 556, 548, 540, 533, 525, 
	518, 510, 503, 496, 489, 482, 475, 468, 
	461, 455, 448, 442, 435, 429, 423, 417
};

/* linear to log pan conversion table (4.2 channel attenuation format) */
static unsigned int pan_table[63] = {
	7959, 7733, 7514, 7301, 7093, 6892, 6697, 6507, 
	6322, 6143, 5968, 5799, 5634, 5475, 5319, 5168, 
	5022, 4879, 4741, 4606, 4475, 4349, 4225, 4105, 
	3989, 3876, 3766, 3659, 3555, 3454, 3356, 3261, 
	3168, 3078, 2991, 2906, 2824, 2744, 2666, 2590, 
	2517, 2445, 2376, 2308, 2243, 2179, 2117, 2057, 
	1999, 1942, 1887, 1833, 1781, 1731, 1682, 1634, 
	1588, 1543, 1499, 1456, 1415, 1375, 1336
};


static void snd_trident_synth_use_inc(trident_t * trident)
{
	MOD_INC_USE_COUNT;
}

static void snd_trident_synth_use_dec(trident_t * trident)
{
	MOD_DEC_USE_COUNT;
}

/*
 * Sample handling operations
 */

static void sample_start(trident_t * trident, snd_trident_voice_t * voice, snd_seq_position_t position);
static void sample_stop(trident_t * trident, snd_trident_voice_t * voice, snd_seq_stop_mode_t mode);
static void sample_freq(trident_t * trident, snd_trident_voice_t * voice, snd_seq_frequency_t freq);
static void sample_volume(trident_t * trident, snd_trident_voice_t * voice, snd_seq_ev_volume * volume);
static void sample_loop(trident_t * trident, snd_trident_voice_t * voice, snd_seq_ev_loop * loop);
static void sample_pos(trident_t * trident, snd_trident_voice_t * voice, snd_seq_position_t position);
static void sample_private1(trident_t * trident, snd_trident_voice_t * voice, unsigned char *data);

static snd_trident_sample_ops_t sample_ops =
{
	sample_start,
	sample_stop,
	sample_freq,
	sample_volume,
	sample_loop,
	sample_pos,
	sample_private1
};

static void snd_trident_simple_init(snd_trident_voice_t * voice)
{
	//voice->handler_wave = interrupt_wave;
	//voice->handler_volume = interrupt_volume;
	//voice->handler_effect = interrupt_effect;
	//voice->volume_change = NULL;
	voice->sample_ops = &sample_ops;
}

static void sample_start(trident_t * trident, snd_trident_voice_t * voice, snd_seq_position_t position)
{
	simple_instrument_t *simple;
	snd_seq_kinstr_t *instr;
	unsigned long flags;
	unsigned int loop_start, loop_end, sample_start, sample_end, start_offset;
	unsigned int value;

	instr = snd_seq_instr_find(trident->synth.ilist, &voice->instr, 0, 1);
	if (instr == NULL)
		return;
	voice->instr = instr->instr;	/* copy ID to speedup aliases */
	simple = KINSTR_DATA(instr);

	spin_lock_irqsave(&trident->reg_lock, flags);
	voice->EC = 0;		/* fixme: do something with envelopes later */
	voice->CTRL = 0;
	voice->Alpha_FMS = 0;

	loop_start = simple->loop_start >> 4;
	if( simple->format & SIMPLE_WAVE_BIDIR) 
		loop_end = 2*(simple->loop_end >> 4) - (simple->loop_start >> 4);
	else
		loop_end = simple->loop_end >> 4;
	sample_start = (simple->start + position) >> 4;
	if( sample_start >= simple->size )
		sample_start = simple->start >> 4;
	sample_end = simple->size;
	start_offset = position >> 4;

	if (simple->format & SIMPLE_WAVE_16BIT)
		voice->CTRL |= 8;
	if (simple->format & SIMPLE_WAVE_STEREO)
		voice->CTRL |= 4;
	if (!(simple->format & SIMPLE_WAVE_UNSIGNED))
		voice->CTRL |= 2;

	if (simple->format & SIMPLE_WAVE_LOOP) {
		voice->CTRL |= 1;
		voice->LBA = virt_to_bus(simple->address.ptr + loop_start);
		if( start_offset >= loop_start ) {
			voice->CSO = start_offset - loop_start;
			voice->negCSO = 0;
		} else {
			voice->CSO = loop_start - start_offset;
			voice->negCSO = 1;
		}
		voice->ESO = loop_end - loop_start - 1;
	} else {
		voice->LBA = virt_to_bus(simple->address.ptr);
		voice->CSO = sample_start;
		voice->ESO = sample_end - 1;
		voice->negCSO = 0;
	}

	if (voice->flags & SND_TRIDENT_VFLG_RUNNING) {
		snd_trident_stop_voice(trident, voice->number);
		voice->flags &= ~SND_TRIDENT_VFLG_RUNNING;
	}

	/* set CSO sign */
	value = inl(TRID_REG(trident, T4D_SIGN_CSO_A));
	if( voice->negCSO ) {
		value |= 1 << (voice->number&31);
	} else {
		value &= ~(1 << (voice->number&31));
	}
	outl(value,TRID_REG(trident, T4D_SIGN_CSO_A));
	
	snd_trident_write_voice_regs(trident, voice->number,
				     voice->LBA, voice->CSO, voice->ESO,
				     voice->Delta, voice->Alpha_FMS,
				     ((voice->FMC & 0x03) << 14) | ((voice->RVol & 0x7f) << 7) | (voice->CVol & 0x7f),
				     voice->GVSel, voice->Pan, voice->Vol,
				     voice->CTRL, voice->EC);
	snd_trident_start_voice(trident, voice->number);
	voice->flags |= SND_TRIDENT_VFLG_RUNNING;
	spin_unlock_irqrestore(&trident->reg_lock, flags);
	snd_seq_instr_free_use(trident->synth.ilist, instr);
}

static void sample_stop(trident_t * trident, snd_trident_voice_t * voice, snd_seq_stop_mode_t mode)
{
	unsigned long flags;

	if (!(voice->flags & SND_TRIDENT_VFLG_RUNNING))
		return;

	switch (mode) {
	default:
		spin_lock_irqsave(&trident->reg_lock, flags);
		snd_trident_stop_voice(trident, voice->number);
		voice->flags &= ~SND_TRIDENT_VFLG_RUNNING;
		spin_unlock_irqrestore(&trident->reg_lock, flags);
		break;
	case SAMPLE_STOP_LOOP:	/* disable loop only */
		voice->CTRL &= ~1;
		spin_lock_irqsave(&trident->reg_lock, flags);
		outb((unsigned char) voice->number, TRID_REG(trident, T4D_LFO_GC_CIR));
		outw((((voice->CTRL << 12) | (voice->EC & 0x0fff)) & 0xffff), CH_GVSEL_PAN_VOL_CTRL_EC);
		spin_unlock_irqrestore(&trident->reg_lock, flags);
		break;
	}
}

static void sample_freq(trident_t * trident, snd_trident_voice_t * voice, snd_seq_frequency_t freq)
{
	unsigned long flags;
	freq >>= 4;

	spin_lock_irqsave(&trident->reg_lock, flags);
	if (freq == 44100)
		voice->Delta = 0xeb3;
	else if (freq == 8000)
		voice->Delta = 0x2ab;
	else if (freq == 48000)
		voice->Delta = 0x1000;
	else
		voice->Delta = (((freq << 12) + freq) / 48000) & 0x0000ffff;

	outb((unsigned char) voice->number, TRID_REG(trident, T4D_LFO_GC_CIR));
	if (trident->device == TRIDENT_DEVICE_ID_NX) {
		outb((unsigned char) voice->Delta, TRID_REG(trident, CH_NX_DELTA_CSO + 3));
		outb((unsigned char) (voice->Delta >> 8), TRID_REG(trident, CH_NX_DELTA_ESO + 3));
	} else {
		outw((unsigned short) voice->Delta, TRID_REG(trident, CH_DX_ESO_DELTA));
	}

	spin_unlock_irqrestore(&trident->reg_lock, flags);
}

static void sample_volume(trident_t * trident, snd_trident_voice_t * voice, snd_seq_ev_volume * volume)
{
	unsigned long flags;
	unsigned short value;

	spin_lock_irqsave(&trident->reg_lock, flags);
	voice->GVSel = 0;	/* use global music volume */
	voice->FMC = 0x03;	/* fixme: can we do something useful with FMC? */
	if (volume->volume >= 0) {
		volume->volume &= 0x3fff;
		/* linear volume -> logarithmic attenuation conversion */
		// TODO: we can probably get extra volume precision if we
		//       also employ the Ec volume envelope register
		for (voice->Vol = 0; voice->Vol < 255; voice->Vol++ ) 
			if (volume->volume >= volume_table[voice->Vol] )
				break;
		voice->RVol = 0;
		voice->CVol = 0;
	}
	if (volume->lr >= 0) {
		volume->lr &= 0x3fff;
		/* approximate linear pan by attenuating channels */
		// TODO: we might be able to get a better approximation by 
		//       scaling channel volume at the same time
		if (volume->lr >= 0x2000) {	/* attenuate left (pan right) */
			value = 0x3fff - volume->lr;
			for (voice->Pan = 0; voice->Pan < 63; voice->Pan++ ) 
				if (value >= pan_table[voice->Pan] )
					break;
		} else {			/* attenuate right (pan left) */
			for (voice->Pan = 0; voice->Pan < 63; voice->Pan++ ) 
				if (volume->lr >= pan_table[voice->Pan] )
					break;
			voice->Pan |= 0x40;
		}
	}
	outb((unsigned char) voice->number, TRID_REG(trident, T4D_LFO_GC_CIR));
	value = (voice->GVSel << 15) | ((voice->Pan & 0x0000007f) << 8) | (voice->Vol & 0x000000ff);
	outw(value, TRID_REG(trident, CH_GVSEL_PAN_VOL_CTRL_EC + 2));
	value = ((voice->FMC & 0x03) << 14) | ((voice->RVol & 0x7f) << 7) | (voice->CVol & 0x7f);
	outw(value, TRID_REG(trident, CH_DX_FMC_RVOL_CVOL));
	spin_unlock_irqrestore(&trident->reg_lock, flags);
}

static void sample_loop(trident_t * trident, snd_trident_voice_t * voice, snd_seq_ev_loop * loop)
{
	unsigned long flags;
	simple_instrument_t *simple;
	snd_seq_kinstr_t *instr;
	unsigned int loop_start, loop_end;

	instr = snd_seq_instr_find(trident->synth.ilist, &voice->instr, 0, 1);
	if (instr == NULL)
		return;
	voice->instr = instr->instr;	/* copy ID to speedup aliases */
	simple = KINSTR_DATA(instr);

	loop_start = simple->loop_start >> 4;
	if( simple->format & SIMPLE_WAVE_BIDIR) 
		loop_end = 2*(simple->loop_end >> 4) - (simple->loop_start >> 4);
	else
		loop_end = simple->loop_end >> 4;

	spin_lock_irqsave(&trident->reg_lock, flags);

	voice->LBA = virt_to_bus(simple->address.ptr + loop_start);
	voice->CSO = 0;
	voice->ESO = loop_end - loop_start - 1;

	outb((unsigned char) voice->number, TRID_REG(trident, T4D_LFO_GC_CIR));
	outb((voice->LBA >> 16), TRID_REG(trident, CH_LBA + 2));
	outw((voice->LBA & 0xffff), TRID_REG(trident, CH_LBA));
	if (trident->device == TRIDENT_DEVICE_ID_NX) {
		outb((voice->ESO >> 16), TRID_REG(trident, CH_NX_DELTA_ESO + 2));
		outw((voice->ESO & 0xffff), TRID_REG(trident, CH_NX_DELTA_ESO));
		outb((voice->CSO >> 16), TRID_REG(trident, CH_NX_DELTA_CSO + 2));
		outw((voice->CSO & 0xffff), TRID_REG(trident, CH_NX_DELTA_CSO));
	} else {
		outw((voice->ESO & 0xffff), TRID_REG(trident, CH_DX_ESO_DELTA + 2));
		outw((voice->CSO & 0xffff), TRID_REG(trident, CH_DX_CSO_ALPHA_FMS + 2));
	}

	spin_unlock_irqrestore(&trident->reg_lock, flags);
	snd_seq_instr_free_use(trident->synth.ilist, instr);
}

static void sample_pos(trident_t * trident, snd_trident_voice_t * voice, snd_seq_position_t position)
{
	unsigned long flags;
	simple_instrument_t *simple;
	snd_seq_kinstr_t *instr;
	unsigned int value;

	instr = snd_seq_instr_find(trident->synth.ilist, &voice->instr, 0, 1);
	if (instr == NULL)
		return;
	voice->instr = instr->instr;	/* copy ID to speedup aliases */
	simple = KINSTR_DATA(instr);

	spin_lock_irqsave(&trident->reg_lock, flags);

	if (simple->format & SIMPLE_WAVE_LOOP) {
		if( position >= simple->loop_start ) {
			voice->CSO = (position - simple->loop_start) >> 4;
			voice->negCSO = 0;
		} else {
			voice->CSO = (simple->loop_start - position) >> 4;
			voice->negCSO = 1;
		}
	} else {
		voice->CSO = position >> 4;
		voice->negCSO = 0;
	}

	/* set CSO sign */
	value = inl(TRID_REG(trident, T4D_SIGN_CSO_A));
	if( voice->negCSO ) {
		value |= 1 << (voice->number&31);
	} else {
		value &= ~(1 << (voice->number&31));
	}
	outl(value,TRID_REG(trident, T4D_SIGN_CSO_A));
	

	outb((unsigned char) voice->number, TRID_REG(trident, T4D_LFO_GC_CIR));
	if (trident->device == TRIDENT_DEVICE_ID_NX) {
		outw((voice->CSO & 0xffff), TRID_REG(trident, CH_NX_DELTA_CSO));
		outb((voice->CSO >> 16), TRID_REG(trident, CH_NX_DELTA_CSO + 2));
	} else {
		outw((voice->CSO & 0xffff), TRID_REG(trident, CH_DX_CSO_ALPHA_FMS) + 2);
	}

	spin_unlock_irqrestore(&trident->reg_lock, flags);
	snd_seq_instr_free_use(trident->synth.ilist, instr);
}

static void sample_private1(trident_t * trident, snd_trident_voice_t * voice, unsigned char *data)
{
}

/*
 * Memory management / sample loading
 */

static int snd_trident_simple_put_sample(void *private_data, simple_instrument_t * instr,
					 char *data, long len, int atomic)
{
	trident_t *trident = snd_magic_cast(trident_t, private_data, -ENXIO);
	unsigned char *block = NULL;
	int size = instr->size;
	int extrasize = 0; 
	int shift = 0;
	int loop_start = instr->loop_start >> 4;
	int loop_end = instr->loop_end >> 4;
	int i;

	if (instr->format & SIMPLE_WAVE_BACKWARD ||
	    instr->format & SIMPLE_WAVE_ULAW) 
		return -EINVAL;	/* not supported */

	if (instr->format & SIMPLE_WAVE_BIDIR) 
		extrasize = (instr->loop_end >> 4) - (instr->loop_start >> 4);

	if (instr->format & SIMPLE_WAVE_16BIT)
		shift++;
	if (instr->format & SIMPLE_WAVE_STEREO)
		shift++;
	size <<= shift;
	extrasize <<= shift;
	loop_start <<= shift;
	loop_end <<= shift;

	if (trident->synth.current_size + size + extrasize > trident->synth.max_size)
		return -ENOMEM;

	if (verify_area(VERIFY_READ, data, size))
		return -EFAULT;

	block = (unsigned char *) snd_kmalloc(size + extrasize, GFP_KERNEL | TRIDENT_GFP_FLAGS);
	if (block == NULL)
		return -ENOMEM;

	copy_from_user(block, data, size);

	if (instr->format & SIMPLE_WAVE_BIDIR) {
		/* move end sample data, if any */
		if (loop_end < size) {
			memmove( block + loop_end + extrasize, 
				 block + loop_end,
				 size - loop_end );
		}
		/* copy each sample...inefficient but works */
		for (i = 0; i < extrasize; i += (1<<shift)) {
			memcpy( block + loop_end + i + 1,
				block + loop_end - i,
				(1<<shift) );
		}
	}

	instr->address.ptr = block;
	trident->synth.current_size += size + extrasize;
	return 0;
}

static int snd_trident_simple_get_sample(void *private_data, simple_instrument_t * instr,
					 char *data, long len, int atomic)
{
	//trident_t *trident = snd_magic_cast(trident_t, private_data, -ENXIO);
	int size = instr->size;
	int shift = 0;
	int loop_end = instr->loop_end >> 4;
	int loop_size = loop_end - (instr->loop_start>>4);

	if (instr->format & SIMPLE_WAVE_16BIT)
		shift++;
	if (instr->format & SIMPLE_WAVE_STEREO)
		shift++;
	size <<= shift;
	loop_end <<= shift;
	loop_size <<= shift;

	if (verify_area(VERIFY_WRITE, data, size))
		return -EFAULT;

	if (instr->format & SIMPLE_WAVE_BIDIR) {
		copy_to_user(data, instr->address.ptr, loop_end);
		if (loop_end < size ) {
			copy_to_user(data+loop_end, instr->address.ptr + loop_end + loop_size, (size - loop_end) );
		}
	} else {
		copy_to_user(data, instr->address.ptr, size);
	}

	return 0;
}

static int snd_trident_simple_remove_sample(void *private_data, simple_instrument_t * instr,
					    int atomic)
{
	trident_t *trident = snd_magic_cast(trident_t, private_data, -ENXIO);
	int size = instr->size;

	snd_kfree(instr->address.ptr);

	if (instr->format & SIMPLE_WAVE_BIDIR) 
		size += (instr->loop_end >> 4) - (instr->loop_start >> 4);
	if (instr->format & SIMPLE_WAVE_16BIT)
		size <<= 1;
	if (instr->format & SIMPLE_WAVE_STEREO)
		size <<= 1;

	trident->synth.current_size -= size;
	if (trident->synth.current_size < 0)	/* shouldnt need this check... */
		trident->synth.current_size = 0;

	return 0;
}

static void select_instrument(trident_t * trident, snd_trident_voice_t * v)
{
	snd_seq_kinstr_t *instr;
	instr = snd_seq_instr_find(trident->synth.ilist, &v->instr, 0, 1);
	if (instr != NULL) {
		if (instr->ops) {
			if (instr->ops->instr_type == snd_seq_simple_id)
				snd_trident_simple_init(v);
		}
		snd_seq_instr_free_use(trident->synth.ilist, instr);
	}
}

/*

 */

static void event_sample(snd_seq_event_t * ev, snd_trident_port_t * p, snd_trident_voice_t * v)
{
	if (v->sample_ops && v->sample_ops->sample_stop)
		v->sample_ops->sample_stop(p->trident, v, SAMPLE_STOP_IMMEDIATELY);
	v->instr.std = ev->data.sample.param.sample.std;
	if (v->instr.std & 0xff000000) {	/* private instrument */
		v->instr.std &= 0x00ffffff;
		v->instr.std |= (unsigned int)ev->source.client << 24;
	}
	v->instr.bank = ev->data.sample.param.sample.bank;
	v->instr.prg = ev->data.sample.param.sample.prg;
	select_instrument(p->trident, v);
}

static void event_cluster(snd_seq_event_t * ev, snd_trident_port_t * p, snd_trident_voice_t * v)
{
	if (v->sample_ops && v->sample_ops->sample_stop)
		v->sample_ops->sample_stop(p->trident, v, SAMPLE_STOP_IMMEDIATELY);
	v->instr.cluster = ev->data.sample.param.cluster.cluster;
	select_instrument(p->trident, v);
}

static void event_start(snd_seq_event_t * ev, snd_trident_port_t * p, snd_trident_voice_t * v)
{
	if (v->sample_ops && v->sample_ops->sample_start)
		v->sample_ops->sample_start(p->trident, v, ev->data.sample.param.position);
}

static void event_stop(snd_seq_event_t * ev, snd_trident_port_t * p, snd_trident_voice_t * v)
{
	if (v->sample_ops && v->sample_ops->sample_stop)
		v->sample_ops->sample_stop(p->trident, v, ev->data.sample.param.stop_mode);
}

static void event_freq(snd_seq_event_t * ev, snd_trident_port_t * p, snd_trident_voice_t * v)
{
	if (v->sample_ops && v->sample_ops->sample_freq)
		v->sample_ops->sample_freq(p->trident, v, ev->data.sample.param.frequency);
}

static void event_volume(snd_seq_event_t * ev, snd_trident_port_t * p, snd_trident_voice_t * v)
{
	if (v->sample_ops && v->sample_ops->sample_volume)
		v->sample_ops->sample_volume(p->trident, v, &ev->data.sample.param.volume);
}

static void event_loop(snd_seq_event_t * ev, snd_trident_port_t * p, snd_trident_voice_t * v)
{
	if (v->sample_ops && v->sample_ops->sample_loop)
		v->sample_ops->sample_loop(p->trident, v, &ev->data.sample.param.loop);
}

static void event_position(snd_seq_event_t * ev, snd_trident_port_t * p, snd_trident_voice_t * v)
{
	if (v->sample_ops && v->sample_ops->sample_pos)
		v->sample_ops->sample_pos(p->trident, v, ev->data.sample.param.position);
}

static void event_private1(snd_seq_event_t * ev, snd_trident_port_t * p, snd_trident_voice_t * v)
{
	if (v->sample_ops && v->sample_ops->sample_private1)
		v->sample_ops->sample_private1(p->trident, v, (unsigned char *) &ev->data.sample.param.raw8);
}

typedef void (trident_sample_event_handler_t) (snd_seq_event_t * ev, snd_trident_port_t * p, snd_trident_voice_t * v);

static trident_sample_event_handler_t *trident_sample_event_handlers[9] =
{
	event_sample,
	event_cluster,
	event_start,
	event_stop,
	event_freq,
	event_volume,
	event_loop,
	event_position,
	event_private1
};

static void snd_trident_sample_event(snd_seq_event_t * ev, snd_trident_port_t * p)
{
	int idx, voice;
	trident_t *trident = p->trident;
	snd_trident_voice_t *v;
	unsigned long flags;

	idx = ev->type - SND_SEQ_EVENT_SAMPLE;
	if (idx < 0 || idx > 8)
		return;
	for (voice = 0; voice < 64; voice++) {
		v = &trident->synth.voices[voice];
		if (v->use && v->client == ev->source.client &&
		    v->port == ev->source.port &&
		    v->index == ev->data.sample.channel) {
			spin_lock_irqsave(&trident->event_lock, flags);
			trident_sample_event_handlers[idx] (ev, p, v);
			spin_unlock_irqrestore(&trident->event_lock, flags);
			return;
		}
	}
}

/*

 */

static void snd_trident_synth_free_voices(trident_t * trident, int client, int port)
{
	int idx;
	snd_trident_voice_t *voice;

	for (idx = 0; idx < 32; idx++) {
		voice = &trident->synth.voices[idx];
		if (voice->use && voice->client == client && voice->port == port)
			snd_trident_free_voice(trident, voice);
	}
}

static int snd_trident_synth_use(void *private_data, snd_seq_port_subscribe_t * info)
{
	snd_trident_port_t *port = (snd_trident_port_t *) private_data;
	trident_t *trident = port->trident;
	snd_trident_voice_t *voice;
	int idx;
	unsigned long flags;

	if (info->synth_voices + info->midi_voices > 32)
		return -EINVAL;
	spin_lock_irqsave(&trident->reg_lock, flags);
	for (idx = 0; idx < info->synth_voices; idx++) {
		voice = snd_trident_alloc_voice(trident, SND_TRIDENT_VOICE_TYPE_SYNTH, info->sender.client, info->sender.port);
		if (voice == NULL) {
			snd_trident_synth_free_voices(trident, info->sender.client, info->sender.port);
			spin_unlock_irqrestore(&trident->reg_lock, flags);
			return -EBUSY;
		}
		voice->index = idx;
		voice->Vol = 0xff;
	}
	for (idx = 0; idx < info->midi_voices; idx++) {
		port->midi_has_voices = 1;
		voice = snd_trident_alloc_voice(trident, SND_TRIDENT_VOICE_TYPE_MIDI, info->sender.client, info->sender.port);
		if (voice == NULL) {
			snd_trident_synth_free_voices(trident, info->sender.client, info->sender.port);
			spin_unlock_irqrestore(&trident->reg_lock, flags);
			return -EBUSY;
		}
		voice->Vol = 0xff;
	}
	if (info->sender.client != SND_SEQ_CLIENT_SYSTEM)
		snd_trident_synth_use_inc(trident);
	spin_unlock_irqrestore(&trident->reg_lock, flags);
	return 0;
}

static int snd_trident_synth_unuse(void *private_data, snd_seq_port_subscribe_t * info)
{
	snd_trident_port_t *port = (snd_trident_port_t *) private_data;
	trident_t *trident = port->trident;
	unsigned long flags;

	spin_lock_irqsave(&trident->reg_lock, flags);
	snd_trident_synth_free_voices(trident, info->sender.client, info->sender.port);
	if (info->sender.client != SND_SEQ_CLIENT_SYSTEM)
		snd_trident_synth_use_dec(trident);
	spin_unlock_irqrestore(&trident->reg_lock, flags);
	return 0;
}

/*

 */

static void snd_trident_synth_free_private_instruments(snd_trident_port_t * p, int client)
{
	snd_seq_instr_free_t ifree;

	memset(&ifree, 0, sizeof(ifree));
	ifree.cmd = SND_SEQ_INSTR_FREE_CMD_PRIVATE;
	snd_seq_instr_list_free_cond(p->trident->synth.ilist, &ifree, client, 0);
}

int snd_trident_synth_event_input(snd_seq_event_t * ev, int direct, void *private_data, int atomic, int hop)
{
	snd_trident_port_t *p = (snd_trident_port_t *) private_data;

	if (p == NULL)
		return -EINVAL;
	if (ev->type >= SND_SEQ_EVENT_SAMPLE &&
	    ev->type <= SND_SEQ_EVENT_SAMPLE_PRIVATE1) {
		snd_trident_sample_event(ev, p);
		return 0;
	}
	if (ev->source.client == SND_SEQ_CLIENT_SYSTEM &&
	    ev->source.port == SND_SEQ_PORT_SYSTEM_ANNOUNCE) {
		if (ev->type == SND_SEQ_EVENT_CLIENT_EXIT) {
			snd_trident_synth_free_private_instruments(p, ev->data.addr.client);
			return 0;
		}
	}
	if (direct) {
		if (ev->type >= SND_SEQ_EVENT_INSTR_BEGIN) {
			snd_seq_instr_event(&p->trident->synth.simple_ops.kops,
					    p->trident->synth.ilist, ev,
					p->trident->synth.seq_client, atomic, hop);
			return 0;
		}
	}
	return 0;
}

static void snd_trident_synth_instr_notify(void *private_data,
					   snd_seq_kinstr_t * instr,
					   int what)
{
	int idx;
	trident_t *trident = snd_magic_cast(trident_t, private_data, );
	snd_trident_voice_t *pvoice;
	unsigned long flags;

	spin_lock_irqsave(&trident->event_lock, flags);
	for (idx = 0; idx < 64; idx++) {
		pvoice = &trident->synth.voices[idx];
		if (pvoice->use && !memcmp(&pvoice->instr, &instr->instr, sizeof(pvoice->instr))) {
			if (pvoice->sample_ops && pvoice->sample_ops->sample_stop) {
				pvoice->sample_ops->sample_stop(trident, pvoice, SAMPLE_STOP_IMMEDIATELY);
			} else {
				snd_trident_stop_voice(trident, pvoice->number);
				pvoice->flags &= ~SND_TRIDENT_VFLG_RUNNING;
			}
		}
	}
	spin_unlock_irqrestore(&trident->event_lock, flags);
}

/*

 */

static void snd_trident_synth_free_port(void *private_data)
{
	snd_trident_port_t *p = (snd_trident_port_t *) private_data;

	if (p)
		snd_midi_channel_free_set(p->chset);
}

static int snd_trident_synth_create_port(trident_t * trident, int idx)
{
	snd_trident_port_t *p;
	snd_seq_port_callback_t callbacks;
	char name[32];
	char *str;
	int result;

	p = &trident->synth.seq_ports[idx];
	p->chset = snd_midi_channel_alloc_set(16);
	if (p->chset == NULL)
		return -ENOMEM;
	p->chset->private_data = p;
	p->trident = trident;
	p->client = trident->synth.seq_client;

	memset(&callbacks, 0, sizeof(callbacks));
	callbacks.use = snd_trident_synth_use;
	callbacks.unuse = snd_trident_synth_unuse;
	callbacks.event_input = snd_trident_synth_event_input;
	callbacks.private_free = snd_trident_synth_free_port;
	callbacks.private_data = p;

	str = "???";
	switch (trident->device) {
	case TRIDENT_DEVICE_ID_DX:	str = "Trident 4DWave-DX"; break;
	case TRIDENT_DEVICE_ID_NX:	str = "Trident 4DWave-NX"; break;
	case TRIDENT_DEVICE_ID_SI7018:	str = "SiS 7018"; break;
	}
	sprintf(name, "%s port %i", str, idx);
	p->chset->port = snd_seq_event_port_attach(trident->synth.seq_client,
						   &callbacks,
						   SND_SEQ_PORT_CAP_WRITE | SND_SEQ_PORT_CAP_SUBS_WRITE,
						   SND_SEQ_PORT_TYPE_MIDI_GENERIC |
						   SND_SEQ_PORT_TYPE_MIDI_GM |
						   SND_SEQ_PORT_TYPE_MIDI_GS |
						   SND_SEQ_PORT_TYPE_DIRECT_SAMPLE |
						   SND_SEQ_PORT_TYPE_SYNTH,
						   name);
	if (p->chset->port < 0) {
		result = p->chset->port;
		snd_trident_synth_free_port(p);
		return result;
	}
	p->port = p->chset->port;
	return 0;
}

/*

 */

static int snd_trident_synth_new_device(snd_seq_device_t *dev)
{
	trident_t *trident;
	int client, i;
	snd_seq_client_callback_t callbacks;
	snd_seq_client_info_t cinfo;
	snd_seq_port_subscribe_t sub;
	snd_simple_ops_t *simpleops;
	char *str;

	trident = *(trident_t **)SND_SEQ_DEVICE_ARGPTR(dev);
	if (trident == NULL)
		return -EINVAL;

	trident->synth.seq_client = -1;

	/* allocate new client */
	memset(&callbacks, 0, sizeof(callbacks));
	callbacks.private_data = trident;
	callbacks.allow_output = callbacks.allow_input = 1;
	client = trident->synth.seq_client =
	    snd_seq_create_kernel_client(trident->card, 1, &callbacks);
	if (client < 0)
		return client;

	/* change name of client */
	memset(&cinfo, 0, sizeof(cinfo));
	cinfo.client = client;
	cinfo.type = KERNEL_CLIENT;
	str = "???";
	switch (trident->device) {
	case TRIDENT_DEVICE_ID_DX:	str = "Trident 4DWave-DX"; break;
	case TRIDENT_DEVICE_ID_NX:	str = "Trident 4DWave-NX"; break;
	case TRIDENT_DEVICE_ID_SI7018:	str = "SiS 7018"; break;
	}
	sprintf(cinfo.name, str);
	strcpy(cinfo.group, SND_SEQ_GROUP_DEVICE);
	snd_seq_kernel_client_ctl(client, SND_SEQ_IOCTL_SET_CLIENT_INFO, &cinfo);

	for (i = 0; i < 4; i++)
		snd_trident_synth_create_port(trident, i);

	trident->synth.ilist = snd_seq_instr_list_new();
	if (trident->synth.ilist == NULL) {
		snd_seq_delete_kernel_client(client);
		trident->synth.seq_client = -1;
		return -ENOMEM;
	}
	trident->synth.ilist->flags = SND_SEQ_INSTR_FLG_DIRECT;

	simpleops = &trident->synth.simple_ops;
	snd_seq_simple_init(simpleops, trident, NULL);
	simpleops->put_sample = snd_trident_simple_put_sample;
	simpleops->get_sample = snd_trident_simple_get_sample;
	simpleops->remove_sample = snd_trident_simple_remove_sample;
	simpleops->notify = snd_trident_synth_instr_notify;

	memset(&sub, 0, sizeof(sub));
	sub.sender.client = SND_SEQ_CLIENT_SYSTEM;
	sub.sender.port = SND_SEQ_PORT_SYSTEM_ANNOUNCE;
	sub.dest.client = client;
	sub.dest.port = 0;
	snd_seq_kernel_client_ctl(client, SND_SEQ_IOCTL_SUBSCRIBE_PORT, &sub);

	return 0;
}

static int snd_trident_synth_delete_device(snd_seq_device_t *dev)
{
	trident_t *trident;

	trident = *(trident_t **)SND_SEQ_DEVICE_ARGPTR(dev);
	if (trident == NULL)
		return -EINVAL;

	if (trident->synth.seq_client >= 0) {
		snd_seq_delete_kernel_client(trident->synth.seq_client);
		trident->synth.seq_client = -1;
	}
	if (trident->synth.ilist)
		snd_seq_instr_list_free(&trident->synth.ilist);
	return 0;
}

static int __init alsa_trident_synth_init(void)
{
	static snd_seq_dev_ops_t ops =
	{
		snd_trident_synth_new_device,
		snd_trident_synth_delete_device
	};

	return snd_seq_device_register_driver(SND_SEQ_DEV_ID_TRIDENT, &ops,
					      sizeof(trident_t*));
}

static void __exit alsa_trident_synth_exit(void)
{
	snd_seq_device_unregister_driver(SND_SEQ_DEV_ID_TRIDENT);
}

module_init(alsa_trident_synth_init)
module_exit(alsa_trident_synth_exit)
MODULE_LICENSE("GPL");
