/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <setjmp.h>
#include <math.h>

#include "cmp.h"
#include "kalc.h"
#include "realfunctions.h"


void f_subtract(void)
{
  /*
   * This function calls the _f_subtract function through the wrapper.
   */

  run2_1_Function(_f_subtract, "-");
}


Object _f_subtract(Object n, Object p, int *err)
{
  /*
   * This function subtracts its arguments.
   */

  switch (type(n)) {
  case TYPE_REAL:
    switch (type(p)) {
    case TYPE_REAL:
      *err = ERR_NOERR;
      n.value.real -= p.value.real;
      break;
    case TYPE_CMP:
      *err = ERR_NOERR;
      __f_reTOc(&n);
      n.value.cmp = cmp_sub(n.value.cmp, p.value.cmp);
      break;
    case TYPE_HXS:
      *err = ERR_NOERR;
      __f_rTOb(&n);
      n.value.h = hxs_sub(n.value.h, p.value.h);
      break;
    default:
      *err = ERR_BADARGUMENTTYPE;
      break;
    }
    break;

  case TYPE_CMP:
    switch (type(p)) {
    case TYPE_REAL:
      __f_reTOc(&p); /* Fall through */
    case TYPE_CMP:
      *err = ERR_NOERR;
      n.value.cmp = cmp_sub(n.value.cmp, p.value.cmp);
      break;
    default:
      *err = ERR_BADARGUMENTTYPE;
      break;
    }
    break;

  case TYPE_HXS:
    switch (type(p)) {
    case TYPE_REAL:
      __f_rTOb(&p); /* Fall through */
    case TYPE_HXS:
      *err = ERR_NOERR;
      n.value.h = hxs_sub(n.value.h, p.value.h);
      break;
    default:
      *err = ERR_BADARGUMENTTYPE;
      break;
    }
    break;
      
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_multiply(void)
{
  /*
   * This function calls the _f_multiply function through the wrapper.
   */

  run2_1_Function(_f_multiply, "*");
}


Object _f_multiply(Object n, Object p, int *err)
{
  /*
   * This function multiplies its arguments.
   */

  switch (type(n)) {
  case TYPE_REAL:
    switch (type(p)) {
    case TYPE_REAL:
      *err = ERR_NOERR;
      n.value.real *= p.value.real;
      break;
    case TYPE_CMP:
      *err = ERR_NOERR;
      __f_reTOc(&n);
      n.value.cmp = cmp_mul(n.value.cmp, p.value.cmp);
      break;
    case TYPE_HXS:
      *err = ERR_NOERR;
      __f_rTOb(&n);
      n.value.h = hxs_mul(n.value.h, p.value.h);
      break;
    default:
      *err = ERR_BADARGUMENTTYPE;
      break;
    }
    break;

  case TYPE_CMP:
    switch (type(p)) {
    case TYPE_REAL:
      __f_reTOc(&p); /* Fall through */
    case TYPE_CMP:
      *err = ERR_NOERR;
      n.value.cmp = cmp_mul(n.value.cmp, p.value.cmp);
      break;
    default:
      *err = ERR_BADARGUMENTTYPE;
      break;
    }
    break;

  case TYPE_HXS:
    switch (type(p)) {
    case TYPE_REAL:
      __f_rTOb(&p); /* Fall through */
    case TYPE_HXS:
      *err = ERR_NOERR;
      n.value.h = hxs_mul(n.value.h, p.value.h);
      break;
    default:
      *err = ERR_BADARGUMENTTYPE;
      break;
    }
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_divide(void)
{
  /*
   * This function calls the _f_divide function through the wrapper.
   */

  run2_1_Function(_f_divide, "/");
}


Object _f_divide(Object n, Object p, int *err)
{
  /*
   * This function divides its two arguments.
   *
   * The only two cases directly handled are two real numbers or two
   * complex numbers. If one argument is real and the other is complex,
   * this function calls itself recursively after transforming the real
   * number into a complex one. This is done because of the checks
   * necessary to do to assure a defined result.
   */

  switch (type(n)) {
  case TYPE_REAL:
    switch (type(p)) {
    case TYPE_REAL:
      *err = ERR_NOERR;
      
      n.value.real /= p.value.real;
      break;
    case TYPE_CMP:
      /* Convert n to complex and call this function recursively */
      n = _f_divide(_f_reTOc(n, err), p, err);
      break;
    case TYPE_HXS:
      *err = ERR_NOERR;
      __f_rTOb(&n);
      n.value.h = hxs_div(n.value.h, p.value.h);
      break;
    default:
      *err = ERR_BADARGUMENTTYPE;
      break;
    }
    break;

  case TYPE_CMP:
    switch(type(p)) {
    case TYPE_REAL:
      /* Convert p to a complex and call this function recursively */
      n = _f_divide(n, _f_reTOc(p, err), err);
      break;
    case TYPE_CMP:
      *err = ERR_NOERR;
      
      n.value.cmp = cmp_div(n.value.cmp, p.value.cmp);
      break;
    default:
      *err = ERR_BADARGUMENTTYPE;
      break;
    }
    break;

  case TYPE_HXS:
    switch (type(p)) {
    case TYPE_REAL:
      __f_rTOb(&p); /* Fall through */
    case TYPE_HXS:
      *err = ERR_NOERR;
      n.value.h = hxs_div(n.value.h, p.value.h);
      break;
    default:
      *err = ERR_BADARGUMENTTYPE;
      break;
    }
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_inv(void)
{
  /*
   * This function calls the _f_inv function through the wrapper.
   */

  run1_1_Function(_f_inv, "inv");
}


Object _f_inv(Object n, int *err)
{
  /*
   * This function returns the reciprocal of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_inv(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    
    n.value.cmp = cmp_inv(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_abs(void)
{
  /*
   * This function calls the _f_abs function through the wrapper.
   */

  run1_1_Function(_f_abs, "abs");
}


Object _f_abs(Object n, int *err)
{
  /*
   * This function returns the absolute value of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_abs(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.type = TYPE_REAL;
    n.value.real = cmp_abs(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_ceil(void)
{
  /*
   * This function runs the _f_ceil function through the wrapper.
   */

  run1_1_Function(_f_ceil, "ceil");
}


Object _f_ceil(Object n, int *err)
{
  /*
   * This function returns the smallest integer greater than or equal
   * to its argument.
   */

  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_ceil(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_floor(void)
{
  /*
   * This function calls the _f_floor function through the wrapper.
   */

  run1_1_Function(_f_floor, "floor");
}


Object _f_floor(Object n, int *err)
{
  /*
   * This function returns the greatest integer less than or equal to
   * its argument.
   */

  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_floor(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;
 
  return n;
}


void f_pctOf(void)
{
  /*
   * This function calls the _f_pctOf function through the wrapper.
   */

  run2_1_Function(_f_pctOf, "%of");
}


Object _f_pctOf(Object n, Object p, int *err)
{
  /*
   * This function returns p percent of n.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
      *err = ERR_NOERR;
      n.value.real = re_pctOf(n.value.real, p.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_pctT(void)
{
  /*
   * This function calls the _f_pctT function through the wrapper.
   */

  run2_1_Function(_f_pctT, "%t");
}


Object _f_pctT(Object n, Object p, int *err)
{
  /*
   * This function returns the percent of p that is represented by n.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
      *err = ERR_NOERR;
      n.value.real = re_pctT(n.value.real, p.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_pctCh(void)
{
  /*
   * This function calls the _f_pctCh function through the wrapper.
   */

  run2_1_Function(_f_pctCh, "%ch");
}


Object _f_pctCh(Object n, Object p, int *err)
{
  /*
   * This function returns the percentage change from p to n as a
   * percentage of p.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
     *err = ERR_NOERR;
      n.value.real = re_pctCh(n.value.real, p.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_mod(void)
{
  /*
   * This function calls the _f_mod function through the wrapper.
   */
  
  run2_1_Function(_f_mod, "mod");
}


Object _f_mod(Object n, Object p, int *err)
{
  /*
   * This function returns n mod p, the way the HP48 does, that is:
   *
   *    a mod b = a - b*floor(a/b)
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    *err = ERR_NOERR;

    n.value.real = re_mod(n.value.real, p.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_gcd(void) 
{
  /*
   * This function calls the _f_gcd function through the wrapper.
   */

  run2_1_Function(_f_gcd, "gcd");
}


Object _f_gcd(Object n, Object p, int *err) 
{
  /*
   * This function returns the greatest common divisor of n and p.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    double u = re_ip(n.value.real);
    double v = re_ip(p.value.real);
    
    *err = ERR_NOERR;

    n.value.real = re_gcd(u, v);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_lcm(void) 
{
  /*
   * This function calls the _f_lcm function through the wrapper.
   */

  run2_1_Function(_f_lcm, "lcm");
}


Object _f_lcm(Object n, Object p, int *err) 
{
  /*
   * This function returns the least common multiple of n and p.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    double u = re_ip(n.value.real);
    double v = re_ip(p.value.real);
    
    *err = ERR_NOERR;

    n.value.real = re_lcm(u, v);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}
